#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string>
#include <vector>
#include "../Serialization.h"
#include "../helper.h"

#if !NO_MAIN || !defined(TEST_ASSERT)
static int iErrors = 0;
static int iChecks = 0;
#endif

#ifndef TEST_ASSERT
# define TEST_ASSERT(...) iChecks++; if (!(__VA_ARGS__)) ++iErrors
#endif

#ifndef TEST_VERIFY
# define TEST_VERIFY(...) do {                                               \
    if (!(__VA_ARGS__)) {                                                    \
        fprintf(stderr, "\n[ERROR] %s() failed:\n\n", __FUNCTION__);         \
        fprintf(stderr, "Violated expectation for this failure was:\n\n  "   \
                #__VA_ARGS__ "\n\n  [%s at line %d]\n\n",                    \
                lastPathComponent(__FILE__).c_str(), __LINE__);              \
    }                                                                        \
    TEST_ASSERT(__VA_ARGS__);                                                \
} while (false);
#endif

#define SRLZ(member) \
    archive->serializeMember(*this, member, #member);

struct SimpleStruct {
    bool a;
    bool b;
    int c;
    float d;
    double e;
    std::string f;

    void serialize(Serialization::Archive* archive) {
        SRLZ(a);
        SRLZ(b);
        SRLZ(c);
        SRLZ(d);
        SRLZ(e);
        SRLZ(f);
    }

    bool operator==(const SimpleStruct& o) const {
        return a == o.a &&
               b == o.b &&
               c == o.c &&
               d == o.d &&
               e == o.e &&
               f == o.f;
    }
};

struct AggregateStruct {
    int one;
    SimpleStruct two;

    void serialize(Serialization::Archive* archive) {
        SRLZ(one);
        SRLZ(two);
    }

    bool operator==(const AggregateStruct& o) const {
        return one == o.one &&
               two == o.two;
    }
};

struct VectorStruct {
    float f;
    std::vector<float> v;

    void serialize(Serialization::Archive* archive) {
        SRLZ(f);
        SRLZ(v);
    }

    bool operator==(const VectorStruct& o) const {
        return f == o.f &&
               v == o.v;
    }
};

struct VectorAggregateStruct {
    int i;
    std::vector<SimpleStruct> v;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(v);
    }

    bool operator==(const VectorAggregateStruct& o) const {
        return i == o.i &&
               v == o.v;
    }
};

struct VectorPtrStruct {
    float f;
    std::vector<double*> v;

    void serialize(Serialization::Archive* archive) {
        SRLZ(f);
        SRLZ(v);
    }

    bool operator==(const VectorPtrStruct& o) const {
        if (f != o.f)
            return false;
        if (v.size() != o.v.size())
            return false;
        for (size_t i = 0; i < v.size(); ++i) {
            if (v[i] && !o.v[i])
                return false;
            if (!v[i] && o.v[i])
                return false;
            if (v[i] && o.v[i] && *v[i] != *o.v[i])
                return false;
        }
        return true;
    }
};

struct AggregateVectorPtrStruct {
    double a;
    double b;
    double c;
    VectorPtrStruct vps;

    void serialize(Serialization::Archive* archive) {
        SRLZ(a);
        SRLZ(b);
        SRLZ(c);
        SRLZ(vps);
    }

    bool operator==(const AggregateVectorPtrStruct& o) const {
        return a == o.a &&
               b == o.b &&
               c == o.c &&
               vps == o.vps;
    }
};

struct SetStruct {
    int i;
    std::set<bool> bs;
    std::set<int> is;
    std::set<unsigned int> uis;
    std::set<float> fs;
    std::set<double> ds;
    std::set<std::string> ss;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(bs);
        SRLZ(is);
        SRLZ(uis);
        SRLZ(fs);
        SRLZ(ds);
        SRLZ(ss);
    }

    bool operator==(const SetStruct& o) const {
        return i == o.i &&
               bs == o.bs &&
               is == o.is &&
               uis == o.uis &&
               fs == o.fs &&
               ds == o.ds &&
               ss == o.ss;
    }
};

struct SetPtrStruct {
    float a;
    float b;
    float c;
    std::set<float*> s;

    void serialize(Serialization::Archive* archive) {
        SRLZ(a);
        SRLZ(b);
        SRLZ(c);
        SRLZ(s);
    }
};

struct MapStruct {
    int i;
    std::map<int,int> iim;
    std::map<int,double> idm;
    std::map<float,int> fim;
    std::map<std::string,bool> sbm;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(iim);
        SRLZ(idm);
        SRLZ(fim);
        SRLZ(sbm);
    }

    bool operator==(const MapStruct& o) const {
        return i == o.i &&
               iim == o.iim &&
               idm == o.idm &&
               fim == o.fim &&
               sbm == o.sbm;
    }
};

struct MapAggregateStruct {
    int i;
    std::map<int,SimpleStruct> m;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(m);
    }

    bool operator==(const MapAggregateStruct& o) const {
        return i == o.i &&
               m == o.m;
    }
};

struct MapPtrStruct {
    float a;
    float b;
    float c;
    float d;
    std::map<float*,float*> m;

    void serialize(Serialization::Archive* archive) {
        SRLZ(a);
        SRLZ(b);
        SRLZ(c);
        SRLZ(d);
        SRLZ(m);
    }
};

struct PolyA {
    int value;

    virtual std::string name() const {
        return "PolyA";
    }

    void serialize(Serialization::Archive* archive) {
        SRLZ(value);
    }
};

struct PolyB : PolyA {
    std::string name() const OVERRIDE {
        return "PolyB";
    }
};

struct PolyC : PolyB {
    std::string name() const OVERRIDE {
        return "PolyC";
    }
};

// manual registration of types PolyB and PolyC required, as these derived types
// did not get into touch with the Serialization framework at compile-time
static Serialization::DataType::NativeDataTypeRegistry<PolyB> _registerPolyB;
static Serialization::DataType::NativeDataTypeRegistry<PolyC> _registerPolyC;

// no manual type registration on this
struct UnregisteredPolyD : PolyC {
    std::string name() const OVERRIDE {
        return "UnregisteredPolyD";
    }
};

// no manual type registration on this
struct UnregisteredPolyE : UnregisteredPolyD {
    std::string name() const OVERRIDE {
        return "UnregisteredPolyE";
    }
};

// simulate type WITHOUT default constructor
struct NoDefaultConstructorPolyF : UnregisteredPolyE {
    NoDefaultConstructorPolyF(int v) {
        this->value = v;
    }

    std::string name() const OVERRIDE {
        return "NoDefaultConstructorPolyF";
    }
};

// simulate type WITHOUT default constructor
struct NoDefaultConstructorPolyG : NoDefaultConstructorPolyF {
    NoDefaultConstructorPolyG(int v) : NoDefaultConstructorPolyF(v) {
    }

    std::string name() const OVERRIDE {
        return "NoDefaultConstructorPolyG";
    }
};

struct VirtPolyA {
    VirtPolyA* p;
    // type size matters! e.g. 'int' type would cause
    // sizeof(VirtPolyA) == sizeof(VirtPolyB) and hence less demanding test
    ssize_t a;

    virtual void serialize(Serialization::Archive* archive) {
        SRLZ(p);
        SRLZ(a);
    }
};

struct VirtPolyB : VirtPolyA {
    ssize_t b;
    using Super = VirtPolyA;

    void serialize(Serialization::Archive* archive) OVERRIDE {
        Super::serialize(archive);
        SRLZ(b);
    }
};

static Serialization::DataType::NativeDataTypeRegistry<VirtPolyB> _registerVirtPolyB;

template<class T>
static Serialization::Archive transfer(const T& src, T& dst) {
    Serialization::RawData raw;

    // serialize
    {
        Serialization::Archive a;
        a.serialize(&src);
        raw = a.rawData();
    }

    // deserialize
    {
        Serialization::Archive a(raw);
        a.deserialize(&dst);
        return a;
    }
}


/////////////////////////////////////////////////////////////////////////////
// The actual test cases ...

static void test_SimpleStruct() {
    SimpleStruct src = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, dst;
    Serialization::Archive a = transfer(src, dst);
    TEST_VERIFY(src == dst);
    src = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "postchange",
    };
    TEST_VERIFY(!(src == dst));
    a.setBoolValue(a.objectByUID(a.rootObject().memberNamed("a").uid()), true);
    a.setBoolValue(a.objectByUID(a.rootObject().memberNamed("b").uid()), false);
    a.setIntValue(a.objectByUID(a.rootObject().memberNamed("c").uid()), 12);
    a.setRealValue(a.objectByUID(a.rootObject().memberNamed("d").uid()), 0.28f);
    a.setRealValue(a.objectByUID(a.rootObject().memberNamed("e").uid()), 188.2);
    a.setStringValue(a.objectByUID(a.rootObject().memberNamed("f").uid()), "postchange");
    a.deserialize(&dst);
    TEST_VERIFY(src == dst);
    src = {
        .a = false,
        .b = false,
        .c = 3127,
        .d = 3.14f,
        .e = 56.25,
        .f = "autotext",
    };
    TEST_VERIFY(!(src == dst));
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("a").uid()), "false");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("b").uid()), "false");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("c").uid()), "3127");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("d").uid()), "3.14");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("e").uid()), "56.25");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("f").uid()), "autotext");
    a.deserialize(&dst);
    TEST_VERIFY(src == dst);
}

static void test_AggregateStruct() {
    AggregateStruct src = {
        .one = 732,
        .two = {
            .a = true,
            .b = false,
            .c = 12,
            .d = 0.28f,
            .e = 188.2,
            .f = "aggregate",
        }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdVector() {
    VectorStruct src = {
        .f = 83.7,
        .v = { 1.0, 7.3, 0.35, 234.17 }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdVectorAggregate() {
    SimpleStruct one = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, two = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "something",
    };
    VectorAggregateStruct src = {
        .i = 1045,
        .v = { one, two }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdSet() {
    SetStruct src = {
        .i = 94783,
        .bs = { false, true, true, false, true },
        .is = { -7, 940, 0, -64 },
        .uis = { 7, 940, 0, 64 },
        .fs = { 3.0f, .3f, 893.54f, 63.924f },
        .ds = { 0.0, 3.345, 193.0, 1349.5479 },
        .ss = { "one", "two", "three", "four" }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdMap() {
    MapStruct src = {
        .i = 6107,
        .iim = {
            { 3, 5 },
            { 0, 9 },
            { 349, 0 }
        },
        .idm = {
            { 0, 5.0 },
            { 9, 0.34 },
            { 20, 120.7 },
            { 40098, 3089.2 }
        },
        .fim = {
            { 9.f, 1 },
            { 0.3f, 2 },
            { 0.34f, 3 },
            { 0.0f, 4 },
            { 22.4f, 5 }
        },
        .sbm = {
            { "one", false },
            { "two", false },
            { "three", true },
            { "four", false },
            { "five", true }
        }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdMapAggregate() {
    SimpleStruct one = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, two = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "something",
    };
    MapAggregateStruct src = {
        .i = 1045,
        .m = {
            { 5, one },
            { 7, two }
        }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_WeakPointers() {
    AggregateVectorPtrStruct src = {
        .a = 1.1,
        .b = 2.2,
        .c = 3.3,
        .vps = {
            .f = 4.4,
            .v = { NULL, NULL }
        }
    }, dst;
    src.vps.v[0] = &src.c;
    src.vps.v[1] = &src.a;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
    TEST_VERIFY(dst.vps.v[0] == &dst.c);
    TEST_VERIFY(dst.vps.v[1] == &dst.a);
}

static void test_StrongPointers() {
    VectorPtrStruct src = {
        .f = 192.1,
        .v = { new double, new double, new double }
    }, dst;
    *src.v[0] = 92.5;
    *src.v[1] = 2.3;
    *src.v[2] = 17.98;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
    TEST_VERIFY(src.v[0] != dst.v[0]);
    TEST_VERIFY(src.v[1] != dst.v[1]);
    TEST_VERIFY(src.v[2] != dst.v[2]);
}

static void test_StrongWeakPointers() {
    struct StrongWeakStruct {
        float* a;
        float* b;

        void serialize(Serialization::Archive* archive) {
            SRLZ(a);
            SRLZ(b);
        }
    } src, dst;

    src.a = src.b = new float(4.1f);

    transfer(src, dst);
    TEST_VERIFY(*src.a == *dst.a);
    TEST_VERIFY(dst.a == dst.b);
}

static void test_SetOfWeakPointers() {
    SetPtrStruct src, dst;

    src = {
        .a = 10.1,
        .b = 11.2,
        .c = 12.3,
    };
    src.s.insert(&src.a);
    src.s.insert(&src.c);

    transfer(src, dst);
    TEST_VERIFY(src.a == dst.a);
    TEST_VERIFY(src.b == dst.b);
    TEST_VERIFY(src.c == dst.c);
    TEST_VERIFY(dst.s.size() == 2);
    TEST_VERIFY(dst.s.count(&dst.a));
    TEST_VERIFY(dst.s.count(&dst.c));
    TEST_VERIFY(src.s != dst.s);
}

static void test_SetOfStrongPointers() {
    SetPtrStruct src, dst;

    src = {
        .a = 10.1,
        .b = 11.2,
        .c = 12.3,
        .s = { new float(13.4), new float(14.5) }
    };

    transfer(src, dst);
    TEST_VERIFY(src.a == dst.a);
    TEST_VERIFY(src.b == dst.b);
    TEST_VERIFY(src.c == dst.c);
    TEST_VERIFY(dst.s.size() == 2);
    TEST_VERIFY(src.s != dst.s);
}

static void test_MapOfWeakPointers() {
    MapPtrStruct src, dst;

    src = {
        .a = 10.1,
        .b = 11.2,
        .c = 12.3,
        .d = 13.4,
    };
    src.m[&src.a] = &src.d;
    src.m[&src.c] = &src.b;

    transfer(src, dst);
    TEST_VERIFY(src.a == dst.a);
    TEST_VERIFY(src.b == dst.b);
    TEST_VERIFY(src.c == dst.c);
    TEST_VERIFY(src.d == dst.d);
    TEST_VERIFY(dst.m.size() == 2);
    TEST_VERIFY(dst.m.count(&dst.a));
    TEST_VERIFY(dst.m.count(&dst.c));
    TEST_VERIFY(src.m != dst.m);
    TEST_VERIFY(dst.m[&dst.a] == &dst.d);
    TEST_VERIFY(dst.m[&dst.c] == &dst.b);
}

static void test_MapOfStrongPointers() {
    MapPtrStruct src, dst;

    src = {
        .a = 10.1,
        .b = 11.2,
        .c = 12.3,
        .d = 13.4,
    };
    src.m[new float(14.5)] = new float(15.6);
    src.m[new float(16.7)] = new float(17.8);

    transfer(src, dst);
    TEST_VERIFY(src.a == dst.a);
    TEST_VERIFY(src.b == dst.b);
    TEST_VERIFY(src.c == dst.c);
    TEST_VERIFY(src.d == dst.d);
    TEST_VERIFY(dst.m.size() == 2);
    TEST_VERIFY(src.m != dst.m);
    for (const auto& it : dst.m) {
        TEST_VERIFY(*it.first == 14.5f || *it.first == 16.7f);
        TEST_VERIFY(*it.second == 15.6f || *it.second == 17.8f);
    }
}

static void test_PolymorphicStrongPointers() {
    struct PolyStrongPtrStruct {
        PolyA* a;
        PolyA* b;
        PolyA* c;
        PolyA* d;

        void serialize(Serialization::Archive* archive) {
            SRLZ(a);
            SRLZ(b);
            SRLZ(c);
            SRLZ(d);
        }
    };

    PolyStrongPtrStruct src = {
        .a = new PolyA,
        .b = NULL,
        .c = new PolyB,
        .d = new PolyC,
    }, dst = { };

    transfer(src, dst);
    TEST_VERIFY(dst.a->name() == "PolyA");
    TEST_VERIFY(dst.b == NULL);
    TEST_VERIFY(dst.c->name() == "PolyB");
    TEST_VERIFY(dst.d->name() == "PolyC");
}

static void test_PolymorphicWeakPointers() {
    struct PolyWeakPtrStruct {
        PolyA polyA;
        UnregisteredPolyD unregisteredPolyD;
        UnregisteredPolyE unregisteredPolyE;
        PolyA* ptrA;
        PolyA* ptrB;
        PolyA* ptrC;
        PolyA* ptrD;

        void serialize(Serialization::Archive* archive) {
            SRLZ(polyA);
            SRLZ(unregisteredPolyD);
            SRLZ(unregisteredPolyE);
            SRLZ(ptrA);
            SRLZ(ptrB);
            SRLZ(ptrC);
            SRLZ(ptrD);
        }
    };

    PolyWeakPtrStruct src = { }, dst = { };

    src.ptrA = &src.polyA;
    src.ptrB = NULL;
    src.ptrC = &src.unregisteredPolyD;
    src.ptrD = &src.unregisteredPolyE;

    transfer(src, dst);
    TEST_VERIFY(dst.ptrA->name() == "PolyA");
    TEST_VERIFY(dst.ptrB == NULL);
    TEST_VERIFY(dst.ptrC->name() == "UnregisteredPolyD");
    TEST_VERIFY(dst.ptrD->name() == "UnregisteredPolyE");
    TEST_VERIFY(dst.ptrA != src.ptrA);
    TEST_VERIFY(dst.ptrC != src.ptrC);
    TEST_VERIFY(dst.ptrD != src.ptrD);
}

static void test_PolymorphicWeakPointersWithoutDefaultConstructor() {
    struct PolyWeakNoDefConstrPtrStruct {
        PolyA polyA;
        NoDefaultConstructorPolyF noDefaultConstructorPolyF;
        NoDefaultConstructorPolyG noDefaultConstructorPolyG;
        PolyA* ptrA;
        PolyA* ptrB;
        PolyA* ptrC;
        PolyA* ptrD;

        PolyWeakNoDefConstrPtrStruct() : noDefaultConstructorPolyF(1), noDefaultConstructorPolyG(2) {
            ptrA = ptrB = ptrC = ptrD = NULL;
        }

        void serialize(Serialization::Archive* archive) {
            SRLZ(polyA);
            SRLZ(noDefaultConstructorPolyF);
            SRLZ(noDefaultConstructorPolyG);
            SRLZ(ptrA);
            SRLZ(ptrB);
            SRLZ(ptrC);
            SRLZ(ptrD);
        }
    };

    PolyWeakNoDefConstrPtrStruct src, dst;

    src.ptrA = &src.polyA;
    src.ptrB = NULL;
    src.ptrC = &src.noDefaultConstructorPolyF;
    src.ptrD = &src.noDefaultConstructorPolyG;

    transfer(src, dst);
    TEST_VERIFY(dst.ptrA->name() == "PolyA");
    TEST_VERIFY(dst.ptrB == NULL);
    TEST_VERIFY(dst.ptrC->name() == "NoDefaultConstructorPolyF");
    TEST_VERIFY(dst.ptrD->name() == "NoDefaultConstructorPolyG");
    TEST_VERIFY(dst.ptrA != src.ptrA);
    TEST_VERIFY(dst.ptrC != src.ptrC);
    TEST_VERIFY(dst.ptrD != src.ptrD);
}

static void test_StdVectorOfPolymorphicStrongPointers() {
    struct VectorPolyPtrStruct {
        std::vector<PolyA*> v;

        void serialize(Serialization::Archive* archive) {
            SRLZ(v);
        }
    };

    VectorPolyPtrStruct src, dst;
    src.v = {
        new PolyA, NULL, new PolyB, new PolyC
    };

    transfer(src, dst);
    TEST_VERIFY(dst.v.size() == 4);
    TEST_VERIFY(dst.v[0]->name() == "PolyA");
    TEST_VERIFY(dst.v[1] == NULL);
    TEST_VERIFY(dst.v[2]->name() == "PolyB");
    TEST_VERIFY(dst.v[3]->name() == "PolyC");
}

static void test_SetOfPolymorphicStrongPointers() {
    struct SetPolyPtrStruct {
        std::set<PolyA*> s;

        void serialize(Serialization::Archive* archive) {
            SRLZ(s);
        }
    };

    SetPolyPtrStruct src, dst;
    src.s = {
        new PolyA, NULL, new PolyB, new PolyC
    };

    transfer(src, dst);

    TEST_VERIFY(dst.s.size() == 4);;
    bool foundPolyA = false;
    bool foundPolyB = false;
    bool foundPolyC = false;
    bool foundNULL  = false;
    for (const PolyA* const & poly : dst.s) {
        if (!poly) {
            foundNULL = true;
        } else if (poly->name() == "PolyA") {
            foundPolyA = true;
        } else if (poly->name() == "PolyB") {
            foundPolyB = true;
        } else if (poly->name() == "PolyC") {
            foundPolyC = true;
        } else {
            TEST_VERIFY(false);
        }
    }
    TEST_VERIFY(foundPolyA);
    TEST_VERIFY(foundPolyB);
    TEST_VERIFY(foundPolyC);
    TEST_VERIFY(foundNULL);
}

static void test_MapValuesOfPolymorphicStrongPointers() {
    struct MapValuePolyPtrStruct {
        std::map<int,PolyA*> m;

        void serialize(Serialization::Archive* archive) {
            SRLZ(m);
        }
    };

    MapValuePolyPtrStruct src, dst;
    src.m[2] = new PolyA;
    src.m[5] = NULL;
    src.m[10] = new PolyB;
    src.m[11] = new PolyC;

    transfer(src, dst);

    TEST_VERIFY(dst.m.size() == 4);
    TEST_VERIFY(src.m[2]->name() == "PolyA");
    TEST_VERIFY(src.m[5] == NULL);
    TEST_VERIFY(src.m[10]->name() == "PolyB");
    TEST_VERIFY(src.m[11]->name() == "PolyC");
}

static void test_MapKeysOfPolymorphicStrongPointers() {
    struct MapKeyPolyPtrStruct {
        std::map<PolyA*,int> m;

        void serialize(Serialization::Archive* archive) {
            SRLZ(m);
        }
    };

    MapKeyPolyPtrStruct src, dst;
    src.m[new PolyA] = 2;
    src.m[NULL] = 5;
    src.m[new PolyB] = 10;
    src.m[new PolyC] = 11;

    transfer(src, dst);

    TEST_VERIFY(dst.m.size() == 4);
    bool foundPolyA = false;
    bool foundPolyB = false;
    bool foundPolyC = false;
    bool foundNULL  = false;
    for (const auto& it : dst.m) {
        if (!it.first) {
            foundNULL = true;
            TEST_VERIFY(it.second == 5);
        } else if (it.first->name() == "PolyA") {
            foundPolyA = true;
            TEST_VERIFY(it.second == 2);
        } else if (it.first->name() == "PolyB") {
            foundPolyB = true;
            TEST_VERIFY(it.second == 10);
        } else if (it.first->name() == "PolyC") {
            foundPolyC = true;
            TEST_VERIFY(it.second == 11);
        } else {
            TEST_VERIFY(false);
        }
    }
    TEST_VERIFY(foundPolyA);
    TEST_VERIFY(foundPolyB);
    TEST_VERIFY(foundPolyC);
    TEST_VERIFY(foundNULL);
}

static void test_overrideSerialize() {
    VirtPolyB src, dst;

    src.p = NULL;
    src.a = 10;
    src.b = 20;

    dst.p = &dst;
    dst.a = -1;
    dst.b = -1;

    transfer(src, dst);

    TEST_VERIFY(dst.p == src.p);
    TEST_VERIFY(dst.a == src.a);
    TEST_VERIFY(dst.b == src.b);
}

static void test_overrideSerializeSelfRef() {
    VirtPolyB src, dst;

    src.p = &src;
    src.a = 10;
    src.b = 20;

    dst.p = NULL;
    dst.a = -1;
    dst.b = -1;

    transfer(src, dst);

    TEST_VERIFY(dst.a == src.a);
    TEST_VERIFY(dst.b == src.b);
    TEST_VERIFY(dst.p != src.p);
    TEST_VERIFY(dst.p == &dst);
}

#if !NO_MAIN

int main() {
    printf("\n");

    int iExceptions = 0;
    try {
        test_SimpleStruct();
        test_AggregateStruct();
        test_StdVector();
        test_StdVectorAggregate();
        test_StdSet();
        test_StdMap();
        test_StdMapAggregate();
        test_WeakPointers();
        test_StrongPointers();
        test_StrongWeakPointers();
        test_SetOfWeakPointers();
        test_SetOfStrongPointers();
        test_MapOfWeakPointers();
        test_MapOfStrongPointers();
        test_PolymorphicStrongPointers();
        test_PolymorphicWeakPointers();
        test_PolymorphicWeakPointersWithoutDefaultConstructor();
        test_StdVectorOfPolymorphicStrongPointers();
        test_SetOfPolymorphicStrongPointers();
        test_MapValuesOfPolymorphicStrongPointers();
        test_MapKeysOfPolymorphicStrongPointers();
        test_overrideSerialize();
        test_overrideSerializeSelfRef();
    } catch (...) {
        iExceptions++;
    }

    if (iErrors || iExceptions)
        printf("\n!!! FAILED !!!  [ There were %d errors, %d exceptions ]\n\n",
               iErrors, iExceptions);
    else
        printf("\nOK (%d checks)\n\n", iChecks);

    return iErrors + iExceptions;
}

#endif // !NO_MAIN
